/////////////////////////////////////////////////////////////////////////////////
//
// License Creative Commons Attribution-NonCommercial-ShareAlike 3.0 Unported License.
//
/////////////////////////////////////////////////////////////////////////////////
//
// This shader is based on Huw Bowles'  FSS,  Fast Adaptive Stationary Sampling,
// shader downloaded from ShaderToy.com and is therfore subject  to  ShaderToy's
// licence conditions, see above. I have slightly modified the shader by
//
//   1) Adapting it to run with the VGHD software.
//
//   2) Replacing the texture based noise function with a purely
//      procedual noise function taken from Alexander Alekseev's
//      Seascape shader, also obtained from ShaderToy.com.
//
//   3) Negating the Y co-ordinate of the vector q  declared  at
//      the start of the main routine. This was done so that the
//      clip's image used as the main texture was upright.
//
//   4) Modifying the main procedure so that the generated smoke
//      or fog overlays the image used as the texture input.
//
//   5) Tidied up the code in a couple of places.
//
//   6) Testing for transparent pixels in the source  image  and
//      passing them through as colour (0,0,0,0,0).
//
// Modification 2 was a lucky accident. I had mistakenly thought that the origin
// was the center of the screen so negating Y would properly flip the image  and
// have no effect on the shader.  However,  the origin is the top left corner of
// the screen and the proper tranformation was y -> 1 - y.  Never-the-less,  the
// incorrect tranformation worked but it also had the side  effect  of  changing
// the appearance of the shader's output from a cloud bank to a smoke cloud.
//
// Modification 6 allows the shader to be used with a background scene.
//
// It has since then been further modified for the Falling Skies - Fishhead Ship
// scene, these modification comprised some general tidying up, minor efficiency
// improvements, a small bug fix and elimination of code that was not needed for
// the particular scene.  This has eliminated some of the previous mods and much
// of the original code. Although the shader now looks very different to the old
// version it is still essentialy the same.
//
/////////////////////////////////////////////////////////////////////////////////

// The following set of parameters control the properties of the cloud  and  can
// be freely modified to suit your particular taste.

uniform vec4 cloudiness;   // Currently only first 3 components used

// CLOUD_DENSITY controls how thick the cloud appears to be.  It can take values
// ranging from 0.0, for no cloud, up to 1.0 for the thickest cloud.

#define CLOUD_DENSITY cloudiness.x

// CLOUD_QUALITY controls the general apearance of the cloud.  It  has no simple
// interpretation but,  if  the shader's internal parameters are left unchanged,
// the following observations can be made
//            <<< THESE NEED TO BE RETESTED SINCE THE MOST RECENT CHANGES >>>
//    0.0 - no internal structure
//    2.0 - internal billows begin to appear
//    3.0 - stronger internal billowing
//    4.0 - internal cloud plumes begin to appear
//    7.0 - stronger internal cloud plumes
//    9.0 - internal plumes begin to break up
//   11.0 - cloud plumes begin to be clumpy
//   13.0 - increased cloud clumpiness
//
// as the values grow even higher the "clumpiness" increases and the size of the
// clumps become smaller. At 100.0 the resmblence to cloud has been lost and the
// effect is more like a snow, but with the snow rising not falling.

#define CLOUD_QUALITY cloudiness.y

// CLOUD_FADE_RATE controls the rate at which the cloud fades as it goes up  the
// screen. A value of 1.0 give linear fading, 2.0 quadratic etc., so high values
// cause the cloud to fade very rapidly leaving clear air above or below a  pool
// of cloud with a distinct boundary while lower layers cause there to be a less
// distinct boundary.

#define CLOUD_FADE_RATE cloudiness.z // Controls rate of fading in Y direction

/////////////////////////////////////////////////////////////////////////////////

// Shader inputs.

uniform float u_Elapsed;    // The elapsed time in seconds
uniform vec2  u_WindowSize; // Window dimensions in pixels

// Use defines here rather than edit the body of the code.

#define iGlobalTime u_Elapsed
#define iResolution u_WindowSize

/////////////////////////////////////////////////////////////////////////////////

// Internal parameters. Only change these if you want to play with the internal
// operation of the shader itself rather than just tweaking the cloud. TheEmu.

#define SAMPLE_COUNT        64
#define DIST_MAX           128.0
#define SUM_LIMIT          0.500
#define SAMPLES_ADAPTIVITY 0.200
#define EPS                0.001

// "Camera" moving in a straight line.

const vec3 lookDir = vec3(-1.0,0.0,0.0);
const vec3 camVel  = vec3(-0.1,0.0,0.0);

// ----------------------------------------------------------------------------

// Constants used in the map function (which is now inlined in raymarch).

const vec4 map_k1 = vec4 ( 0.25000, 0.12500, 0.06250, 0.03125 ) * 3.0;
      vec4 map_k2 = vec4 ( 1.0, 2.02, 4.10, 8.24 ) * CLOUD_QUALITY;
const vec3 map_k3 = vec3 ( 1.1500, 1.0925, 0.9200 );
const vec3 map_k4 = vec3 ( 0.7000, 0.7000, 0.7000 );

// ----------------------------------------------------------------------------

// The macro spacing calculates the sample spacing for our sample count.

const float spacing_kk = log ( SAMPLES_ADAPTIVITY * DIST_MAX + 1.0 )
                           / ( SAMPLES_ADAPTIVITY * float(SAMPLE_COUNT) );

#define spacing(t) ( ( SAMPLES_ADAPTIVITY*max(t,0.0) + 1.0 ) * spacing_kk )

// ----------------------------------------------------------------------------

// The hash macro generates a random scalar floating point value  in  the
// range 0.0 to 1.0. Its input, q, is a value of type vec2.

#define hash(q) (fract(sin(dot(q,vec2(127.1,311.7)))*43758.5453123)*2.0-1.0)

// ----------------------------------------------------------------------------

vec4 raymarch ( in vec3 ro, in vec3 rd )
 {
   // Setup sampling

   float dt = exp2 ( floor ( log2(spacing(0.0)) ) );
   float tt = 0.0;
   float wt = 1.0;

   vec4 sum = vec4 ( 0.0 );

   // SAMPLE_COUNT can be quite large without unduly affecting
   // the shader's performance because the following loop will
   // normaly exit when sum.a exceeds the threshold SUM_LIMIT.

   for ( int i=0; (i<SAMPLE_COUNT) && (sum.a<SUM_LIMIT); i++ )
    {
      vec3 p = ro + tt*rd;

      // Evaluate the map function to get the colour contribution.
      // Both the map and the noise function that it  called  have 
      // been inlined here and rewritten to be more parallelisable
      // than the original version. I have used macros to keep the
      // code neat and,  I hope,  more easily readable than it was
      // without them while still completely inlining both the map
      // function and the noise function.

      #define n00 vec2 ( 0.0, 0.0 )
      #define n01 vec2 ( 0.0, 1.0 )
      #define n10 vec2 ( 1.0, 0.0 )
      #define n11 vec2 ( 1.0, 1.0 )

      #define PP(ix) p.xz * map_k2[ix]
      #define FF(ix) fract ( pp[ix] )
      #define II(ix) floor ( pp[ix] )
      #define UU(ix) ff[ix]*ff[ix]*(3.0-2.0*ff[ix])
      #define NA(ix) vec2 ( hash(ii[ix]+n00), hash(ii[ix]+n10) )
      #define NB(ix) vec2 ( hash(ii[ix]+n01), hash(ii[ix]+n11) )
      #define AB(ix) mix ( NA(ix), NB(ix), uu[ix].xx )
      #define QQ(ix) mix ( ab[ix].x, ab[ix].y, uu[ix].y )

      vec2 pp[4] = vec2[4] ( PP(0), PP(1), PP(2), PP(3) );
      vec2 ff[4] = vec2[4] ( FF(0), FF(1), FF(2), FF(3) );
      vec2 ii[4] = vec2[4] ( II(0), II(1), II(2), II(3) );
      vec2 uu[4] = vec2[4] ( UU(0), UU(1), UU(2), UU(3) );
      vec2 ab[4] = vec2[4] ( AB(0), AB(1), AB(2), AB(3) );
      vec4 qq    = vec4    ( QQ(0), QQ(1), QQ(2), QQ(3) );

      float ccc = clamp ( dot(map_k1,qq) - p.y, 0.0, 1.0 );
      vec4  col = vec4 ( mix(map_k3,map_k4,ccc)*ccc, ccc );

      // Integrate samples.

      sum += dt * wt * col;

      // Next sample.  The step, dt, will be halved, doubled
      // or left unchanged depending on the value of ss. The
      // use of smoothstep in place of step prevents  sudden
      // transitions and hopefully any artifacts that  might
      // arise from their presence. The sample weight factor
      // has terms for the sample size and distance fading.

      float ss = spacing(tt) / dt;

//    dt *= 0.5 + step(1.0,ss)*0.5 + step(2.0,ss);
      dt *= 0.5 + smoothstep(0.5,1.5,ss)*0.5 + smoothstep(1.5,2.5,ss);

      tt += dt;

      wt = ss * (1.0-sum.a)
         * ( 1.0 - clamp((tt/(DIST_MAX)-0.85)*6.66666,0.0,1.0) );

    }

   sum.rgb /= max(sum.a,0.00001);

   return sum;

 }

// ----------------------------------------------------------------------------

void main(void)
 { 
   // "Camera" for the cloud generation.

   vec2 pp = (gl_FragCoord.xy/iResolution.xy)*2.0 - 1.0;
   vec3 ro = vec3(0.0,1.9,0.0) + mod(iGlobalTime*camVel,10.0);
   vec3 ww = normalize ( lookDir );
   vec3 uu = cross ( vec3(0.0,1.0,0.0), ww );
   vec3 vv = cross ( ww, uu );
   vec3 rd = normalize ( pp.x*uu + 1.2*pp.y*vv + 1.5*ww );

   // Generate the cloud using Huw Bowles's FSS method.

   vec4 rr = clamp ( raymarch(ro,rd), 0.0, 1.0 );
   vec4 cc = vec4 ( mix(vec3(1.0),rr.rgb,rr.a), 0.7 );

   gl_FragColor = cc * gl_Color;

 }

// ----------------------------------------------------------------------------
